/* Direktno implementiramo hijerarhiju ustanovljenu u parseru. 
 *
 * BITNO:
 * prilikom definisanja tipova cvorova sintaksnog stabla treba da sagledamo koliko razlicitih tipova rezultata akcija
 * imamo i da svakom tipu rezultata dodelimo poseban tip cvora sintaksnog stabla. Da bismo pravilno definisali 
 * hijerarhiju klasa za predstavljanje cvorova moramo pazljivo da pogledamo gramatiku koja opisuje nas jezik 
 * matematickih funkcija. Da bismo definisali hijerarhiju cvorova zgodno je da krenemo od dna gramatike:
 * Node ---> ConstantNode (moze se objediniti i sa konstantnim funkcijama)
 *      ---> ConstantFunctionNode (moze se objediniti sa onim iznad u isti tip)
 *      ---> IdentifierNode (funkcije predstavljene identifikatorima)
 *      ---> CompositionNode (kompozicija postojece promenljive sa drugom funkcijom)
 *      ---> IdentityNode   (identicka funkcija)
 *      ---> ValueNode      (vrednost funkcije u tacki)
 *      ---> DiffNode       (izvod funkcije)
 *      ---> ElementarnFunkcija ---> SinNode
 *                              ---> CosNode
 *                              ---> TanNode
 *                              ---> CotNode
 *                              ---> LogNode
 *                              ---> ExpNode
 *                              ---> SqrtNode
 *                              ---> PowNode
 *                              ---> SuprotniNode
 *      ---> BinaryNode (binarne operacije. ne mora se dalje specijalizovati, ali moze. stvar izbora)
 *      ---> EmptyNode          (prazna naredba)
 *      ---> PrintNode          (stampanje izraza)
 *      ---> AssignemntNode     (naredba dodele)
 *      ---> SequenceNode       (nizanje naredbi)
 * BITNO:
 * Cvor kao svoje atribute moze sadrzati samo elementarne vrednosti, a ne vrednosti koji su rezultat
 * izvrsavanja nekog podstabla. Ako tekuci cvor zavisi od nekog podstabla, onda kao svoj atribut
 * mora sadrzati koren onog podstabla ili stabala od kojih zavisi. 
 */

#ifndef SYNTAX_TREE_H
#define SYNTAX_TREE_H

#include <iostream>
#include <cstdlib>
#include <vector>
#include <exception>

#include "FuncSymbolTable.hpp"
#include "Funkcije.hpp"

/* bazna klasa hijerarhije */
class SyntaxTreeNode {
public:
    /* virtuelni destruktor da bismo osigurali nadovezivanje destruktora, prilikom unistavanja objekata */
    virtual ~SyntaxTreeNode() {}

    /* metod koji stampa cvor - prevodi ga nazad u kod */
    virtual void Print(std::ostream& s) const = 0;
    /* metod koji izvrsava naredbu koju cvor predstavlja 
     * naredba moze zavisiti od promenljivih iz tablice simbola ili kreirati nove promenljive
     * zbog toga kao argument treba da prosledimo referencu na tablicu. s obzirom da naredba moze
     * menjati sadrzaj tablice, tablicu ne smemo da prosledimo kao const.
     * BITNO:
     * obratite paznju na rezultat interpretacije cvora. 
     * rezultat interpretacije je nova funkcija. 
     */
    virtual Funkcija* Interpret(FuncSymbolTable& fs) const = 0;
    /* metod racuna vrednost cvora u tacki x
     * kao argumente moramo da prosledimo tablicu simbola i tacku u kojoj racunamo vrednost cvora
     */
    virtual double Calculate(FuncSymbolTable& fs, double x) const = 0;
    /* metod pravi kopiju cvora */
    virtual SyntaxTreeNode* Clone() const = 0;
};

/* radi lakseg stampanja preopterecujemo operator << */
std::ostream& operator <<(std::ostream& s, const SyntaxTreeNode& st);

/* izvodimo novi tip cvora koji predstavlja konstante */
class ConstantNode : public SyntaxTreeNode {

private:
    /* u cvoru cuvamo vrednost od koje zavisi */
    double _value;
public:
    /* klasa je konkretna, pa nam treba eksplicitni konstruktor */
    ConstantNode(double val);
    /* klasa ima atribute prostih tipova, pa mozemo i da ne implemetiramo
     * eksplicitni konstruktor kopije. podrazumevani konstruktor kopije bi radio isto
     * ono sto radi i ovaj eksplicitni, a to je prosto kopiranje vrednosti.
     */
    ConstantNode(const ConstantNode& cn);

    /* nasledjene metode */
    virtual void Print(std::ostream& s) const;
    virtual Funkcija* Interpret(FuncSymbolTable& fs) const;
    virtual double Calculate(FuncSymbolTable& fs, double x) const;
    virtual SyntaxTreeNode* Clone() const;
};

/* izvodimo novi tip cvora koji predstavlja konstante funkcije. moze se izjednaciti sa
 * ConstantNode. jer predstavlja potpuno istu stvar. 
 */
class ConstantFunctionNode : public SyntaxTreeNode {

private:
    /* u cvoru cuvamo vrednost od koje zavisi */
    double _value;
public:
    /* klasa je konkretna, pa nam treba eksplicitni konstruktor */
    ConstantFunctionNode(double val);
    /* klasa ima atribute prostih tipova, pa mozemo i da ne implemetiramo
     * eksplicitni konstruktor kopije. podrazumevani konstruktor kopije bi radio isto
     * ono sto radi i ovaj eksplicitni, a to je prosto kopiranje vrednosti.
     */
    ConstantFunctionNode(const ConstantFunctionNode& cn);

    /* nasledjene metode */
    virtual void Print(std::ostream& s) const;
    virtual Funkcija* Interpret(FuncSymbolTable& fs) const;
    virtual double Calculate(FuncSymbolTable& fs, double x) const;
    virtual SyntaxTreeNode* Clone() const;
};

/* izdvajamo novi tip koji predstavlja identifikatore */
class IdentifierNode : public SyntaxTreeNode {

private:
    std::string _name;
public:
    IdentifierNode(const std::string& s);
    IdentifierNode(const IdentifierNode& cn);

    /* nasledjene metode */
    virtual void Print(std::ostream& s) const;
    virtual Funkcija* Interpret(FuncSymbolTable& fs) const;
    virtual double Calculate(FuncSymbolTable& fs, double x) const;
    virtual SyntaxTreeNode* Clone() const;
};

/* izdvajamo novi tip koji predstavlja identicku funkciju 
 * BITNO:
 * iako ova klasa kao i mnoge druge nema svojih atributa, ne smemo ih tek tako
 * objediniti, jer su rezultati interpretacije razliciti. razlicit rezultat interpretacije
 * zahteva razlicit tip podataka.
 */
class IdentityNode : public SyntaxTreeNode {

public:

    /* nasledjene metode */
    virtual void Print(std::ostream& s) const;
    virtual Funkcija* Interpret(FuncSymbolTable& fs) const;
    virtual double Calculate(FuncSymbolTable& fs, double x) const;
    virtual SyntaxTreeNode* Clone() const;
};

/* kreiramo novi tip koji predstavlja funkcijski cvor u opstem slucaju
 * radi se o cvoru cija rezultat izrvrsavanja zavisi od jednog argumenta
 * taj argument moze biti bilo koji izraz, dakle neko podstablo
 * 
 * funkcijski cvor je bazna klasa nove hijerarhije klasa. 
 */
class FunctionNode : public SyntaxTreeNode {

/* radi se o hijerarhiji, pa atribut mora biti protected da bi bio vidljiv
 * izvedenim klasama 
 */
protected:
    /* atribut je koren nekog podstabla, jer u opstem slucaju izraz (argument) od kojeg zavisi 
     * nasa funkcija moze biti bilo sta.
     *
     * BITNO:
     * cesta greska je da na ovom mestu napisemo funkcija. takva greska nece dovesti do problema u radu
     * programa, ali je sustinski pogresna, jer ne vodi izgradnji sintaksnog stabla. Sintaksno stablo je
     * sacinjeno od cvorova, ciji su rezultati interpretacije funkcije. Dakle, funkcija ne moze biti atribut
     * klasa, vec samo rezultat interpretacije.
     * Problem se krije u tome sto ovoga moramo biti svesni unapred. Ako bismo napravili ovakvu gresku, program bi 
     * vrlo verovatno radio, ali ne bi kreirao sintaksno stablo, vec samo novi omotac oko klase Funkcija, bez ikakve 
     * potrebe. 
     */
    SyntaxTreeNode* _expr;

public:
    /* ekplicitni konstruktor je neophodan da bismo inicijalizovali atributa */
    FunctionNode(SyntaxTreeNode * expr);
    /* eksplicitni destruktor, jer u klasi cuvamo dinamicke resurse */
    ~FunctionNode();

    /* nasledjene metode */
    virtual void Print(std::ostream& s) const = 0;
    virtual Funkcija* Interpret(FuncSymbolTable& fs) const = 0;
    virtual double Calculate(FuncSymbolTable& fs, double x) const = 0;
    virtual SyntaxTreeNode* Clone() const = 0;
};

/* izdvajamo novi tip cvora koji predstavlja kompoziciju funkcija */
class CompositionNode : public FunctionNode {

private:
    /* pored nasledjenog atributa, treba nam i identifikator funkcije */
    std::string _name;
public:
    /* eksplicitni konstruktor */
    CompositionNode(const std::string& s, SyntaxTreeNode* expr);
    /* eksplicitni konstruktor kopije je neophodan da bismo kreirali duboku kopiju atributa */
    CompositionNode(const CompositionNode& cn);

    /* nasledjene metode */
    virtual void Print(std::ostream& s) const;
    virtual Funkcija* Interpret(FuncSymbolTable& fs) const;
    virtual double Calculate(FuncSymbolTable& fs, double x) const;
    virtual SyntaxTreeNode* Clone() const;
};

/* izdvajamo novi tip koji predstavlja stepenovanje */
class PowNode : public FunctionNode {

/* da bismo stepenovali izraz, treba nam eksponnet */
private:
    /* u opstem slucaju, vrednost stepena dobijamo interpretiranjem nekog podstabla */
    SyntaxTreeNode* _stepen;
public:
    /* eksplicitni konstruktor */
    PowNode(SyntaxTreeNode* expr, SyntaxTreeNode* step);
    /* eksplicitni konstruktor kopije je neophodan da bismo kreirali duboku kopiju atributa */
    PowNode(const PowNode& cn);
    /* neophodan nam je eksplicitni destruktor kojim unistavamo atribut klase */
    ~PowNode();

    /* nasledjene metode */
    virtual void Print(std::ostream& s) const;
    virtual Funkcija* Interpret(FuncSymbolTable& fs) const;
    virtual double Calculate(FuncSymbolTable& fs, double x) const;
    virtual SyntaxTreeNode* Clone() const;
};

/* izdvajamo novi tip koji predstavlja sin funkciju */
class SinNode : public FunctionNode {

/* klasa nema novih atributa izvan nasledjenog */

public:
    /* eksplicitni konstruktor */
    SinNode(SyntaxTreeNode* expr);
    /* eksplicitni konstruktor kopije je neophodan da bismo kreirali duboku kopiju atributa */
    SinNode(const SinNode& cn);

    /* nasledjene metode */
    virtual void Print(std::ostream& s) const;
    virtual Funkcija* Interpret(FuncSymbolTable& fs) const;
    virtual double Calculate(FuncSymbolTable& fs, double x) const;
    virtual SyntaxTreeNode* Clone() const;
};

/* izdvajamo novi tip koji predstavlja cos funkciju */
class CosNode : public FunctionNode {

/* klasa nema novih atributa izvan nasledjenog */

public:
    /* eksplicitni konstruktor */
    CosNode(SyntaxTreeNode* expr);
    /* eksplicitni konstruktor kopije je neophodan da bismo kreirali duboku kopiju atributa */
    CosNode(const CosNode& cn);

    /* nasledjene metode */
    virtual void Print(std::ostream& s) const;
    virtual Funkcija* Interpret(FuncSymbolTable& fs) const;
    virtual double Calculate(FuncSymbolTable& fs, double x) const;
    virtual SyntaxTreeNode* Clone() const;
};

/* izdvajamo novi tip koji predstavlja tan funkciju */
class TanNode : public FunctionNode {

/* klasa nema novih atributa izvan nasledjenog */

public:
    /* eksplicitni konstruktor */
    TanNode(SyntaxTreeNode* expr);
    /* eksplicitni konstruktor kopije je neophodan da bismo kreirali duboku kopiju atributa */
    TanNode(const TanNode& cn);

    /* nasledjene metode */
    virtual void Print(std::ostream& s) const;
    virtual Funkcija* Interpret(FuncSymbolTable& fs) const;
    virtual double Calculate(FuncSymbolTable& fs, double x) const;
    virtual SyntaxTreeNode* Clone() const;
};

/* izdvajamo novi tip koji predstavlja cot funkciju */
class CotNode : public FunctionNode {

/* klasa nema novih atributa izvan nasledjenog */

public:
    /* eksplicitni konstruktor */
    CotNode(SyntaxTreeNode* expr);
    /* eksplicitni konstruktor kopije je neophodan da bismo kreirali duboku kopiju atributa */
    CotNode(const CotNode& cn);

    /* nasledjene metode */
    virtual void Print(std::ostream& s) const;
    virtual Funkcija* Interpret(FuncSymbolTable& fs) const;
    virtual double Calculate(FuncSymbolTable& fs, double x) const;
    virtual SyntaxTreeNode* Clone() const;
};

/* izdvajamo novi tip koji predstavlja log funkciju */
class LogNode : public FunctionNode {

/* klasa nema novih atributa izvan nasledjenog */

public:
    /* eksplicitni konstruktor */
    LogNode(SyntaxTreeNode* expr);
    /* eksplicitni konstruktor kopije je neophodan da bismo kreirali duboku kopiju atributa */
    LogNode(const LogNode& cn);

    /* nasledjene metode */
    virtual void Print(std::ostream& s) const;
    virtual Funkcija* Interpret(FuncSymbolTable& fs) const;
    virtual double Calculate(FuncSymbolTable& fs, double x) const;
    virtual SyntaxTreeNode* Clone() const;
};

/* izdvajamo novi tip koji predstavlja exp funkciju */
class ExpNode : public FunctionNode {

/* klasa nema novih atributa izvan nasledjenog */

public:
    /* eksplicitni konstruktor */
    ExpNode(SyntaxTreeNode* expr);
    /* eksplicitni konstruktor kopije je neophodan da bismo kreirali duboku kopiju atributa */
    ExpNode(const ExpNode& cn);

    /* nasledjene metode */
    virtual void Print(std::ostream& s) const;
    virtual Funkcija* Interpret(FuncSymbolTable& fs) const;
    virtual double Calculate(FuncSymbolTable& fs, double x) const;
    virtual SyntaxTreeNode* Clone() const;
};

/* izdvajamo novi tip koji predstavlja sqrt funkciju */
class SqrtNode : public FunctionNode {

/* klasa nema novih atributa izvan nasledjenog */

public:
    /* eksplicitni konstruktor */
    SqrtNode(SyntaxTreeNode* expr);
    /* eksplicitni konstruktor kopije je neophodan da bismo kreirali duboku kopiju atributa */
    SqrtNode(const SqrtNode& cn);

    /* nasledjene metode */
    virtual void Print(std::ostream& s) const;
    virtual Funkcija* Interpret(FuncSymbolTable& fs) const;
    virtual double Calculate(FuncSymbolTable& fs, double x) const;
    virtual SyntaxTreeNode* Clone() const;
};

/* izdvajamo novi tip koji predstavlja unarni minus */
class SuprotniNode : public FunctionNode {

/* klasa nema novih atributa izvan nasledjenog */

public:
    /* eksplicitni konstruktor */
    SuprotniNode(SyntaxTreeNode* expr);
    /* eksplicitni konstruktor kopije je neophodan da bismo kreirali duboku kopiju atributa */
    SuprotniNode(const SuprotniNode& cn);

    /* nasledjene metode */
    virtual void Print(std::ostream& s) const;
    virtual Funkcija* Interpret(FuncSymbolTable& fs) const;
    virtual double Calculate(FuncSymbolTable& fs, double x) const;
    virtual SyntaxTreeNode* Clone() const; 
};

/* izdvajamo novi tip koji predstavlja izvod funkcije */
class DiffNode : public FunctionNode {

/* klasa nema novih atributa izvan nasledjenog */

public:
    /* eksplicitni konstruktor */
    DiffNode(SyntaxTreeNode* expr);
    /* eksplicitni konstruktor kopije je neophodan da bismo kreirali duboku kopiju atributa */
    DiffNode(const DiffNode& cn);

    /* nasledjene metode */
    virtual void Print(std::ostream& s) const;
    virtual Funkcija* Interpret(FuncSymbolTable& fs) const;
    virtual double Calculate(FuncSymbolTable& fs, double x) const;
    virtual SyntaxTreeNode* Clone() const; 
};

/* izdvajamo novi tip koji predstavlja izracunavanje vrednosti u tacki */
class ValueNode : public FunctionNode {

/* da bismo izracunali vrednost funkcije u tacki, treba nam tacka */
private:
    /* u opstem slucaju, vrednost tacke dobijamo interpretiranjem nekog podstabla */
    SyntaxTreeNode* _x;

public:
    /* eksplicitni konstruktor */
    ValueNode(SyntaxTreeNode* expr, SyntaxTreeNode* x);
    /* eksplicitni konstruktor kopije je neophodan da bismo kreirali duboku kopiju atributa */
    ValueNode(const ValueNode& cn);
    /* neophodan nam je eksplicitni destruktor kojim unistavamo atribut klase */
    ~ValueNode();

    /* nasledjene metode */
    virtual void Print(std::ostream& s) const;
    virtual Funkcija* Interpret(FuncSymbolTable& fs) const;
    virtual double Calculate(FuncSymbolTable& fs, double x) const;
    virtual SyntaxTreeNode* Clone() const; 
};

/* izdvajamo novi tip koji predstavlja binarnu operaciju 
 * BITNO:
 * u implementaciji cemo malo varati, tako sto cemo sve operacije objediniti u jedan cvor
 * i prilikom interpretacije cemo ih razlikovati na osnovu simbola. resenje koje je ekspresivnije
 * bi ovaj tip uzelo kao bazni i onda daljom specijalizacijom izvelo plus, minus, puta i podeljeno
 * cvorove. 
 */
class BinaryNode : public SyntaxTreeNode {

/* binarna operacija zavisi od svog levog i desnog operanda koji su opstem slucaju
 * podstabla. takodje, cuvacemo i simbol operacije kao atribut da bismo razlikovali
 * operacije. 
 */
private:
    
    SyntaxTreeNode* _left;
    SyntaxTreeNode* _right;
    std::string _symbol;
public:
    /* eksplicitni konstruktor */
    BinaryNode(const std::string _s, SyntaxTreeNode* l, SyntaxTreeNode* r);
    /* eksplicitni konstruktor kopije je neophodan da bismo kreirali duboku kopiju atributa */
    BinaryNode(const BinaryNode& bo);
    /* neophodan nam je eksplicitni destruktor kojim unistavamo atribute klase */
    ~BinaryNode();

    /* nasledjene metode */
    virtual void Print(std::ostream& s) const;
    virtual Funkcija* Interpret(FuncSymbolTable& fs) const;
    virtual double Calculate(FuncSymbolTable& fs, double x) const;
    virtual SyntaxTreeNode* Clone() const; 
};

/* izdvajamo novi tip koji predstavja operaciju dodele 
 * BITNO:
 * primetimo da u cvoru kao atribute cuvamo ono od cega zavisi operacija dodele
 * stvarno dodeljivanje vrednosti datoj promenljivoj se desava tek u interpretaciji datog cvora.
 */
class AssignmentNode : public SyntaxTreeNode {

/* kao atribute klase cuvamo naziv promenljive i koren podstabla koje
 * predstavlja desnu stranu operacije dodele. 
 */
private:
    std::string _name;
    SyntaxTreeNode* _expr;
public:
    /* eksplicitni konstruktor */
    AssignmentNode(const std::string& s, SyntaxTreeNode* expr);
    /* eksplicitni konstruktor kopije je neophodan da bismo kreirali duboku kopiju atributa */
    AssignmentNode(const AssignmentNode& an);
    /* neophodan nam je eksplicitni destruktor kojim unistavamo atribute klase */
    ~AssignmentNode();

    /* nasledjene metode */
    virtual void Print(std::ostream& s) const;
    virtual Funkcija* Interpret(FuncSymbolTable& fs) const;
    virtual double Calculate(FuncSymbolTable& fs, double x) const;
    virtual SyntaxTreeNode* Clone() const; 
};

/* izdvajam novi tip koji predstavlja naredbu stampanja */
class PrintNode : public SyntaxTreeNode {

/* kao atribut klase cuvamo koren podstabla koji predstavlja dati izraz koji se stampa*/
private:
    SyntaxTreeNode* _expr;
public:
    /* eksplicitni konstruktor */
    PrintNode(SyntaxTreeNode* expr);
    /* eksplicitni konstruktor kopije je neophodan da bismo kreirali duboku kopiju atributa */
    PrintNode(const PrintNode& an);
    /* neophodan nam je eksplicitni destruktor kojim unistavamo atribute klase */
    ~PrintNode();

    /* nasledjene metode */
    virtual void Print(std::ostream& s) const;
    virtual Funkcija* Interpret(FuncSymbolTable& fs) const;
    virtual double Calculate(FuncSymbolTable& fs, double x) const;
    virtual SyntaxTreeNode* Clone() const; 
};

/* izdvajamo novi tip koji predstavlja niz naredbi */
class SequenceNode : public SyntaxTreeNode{

/* kao atribut klase cuvamo niz (vektor) naredbi */
private:
    /* primetimo da u vektoru cuvamo pokazivace da bismo osigurali polimorfno ponasanje */
    std::vector<SyntaxTreeNode*> _statements;
public:
    /* eksplicitni konstruktor */
    SequenceNode() {}
    /* eksplicitni konstruktor kopije je neophodan da bismo kreirali duboku kopiju atributa */
    SequenceNode(const SequenceNode& sn);
    /* neophodan nam je eksplicitni destruktor kojim unistavamo atribute klase */
    ~SequenceNode();

    /* pomocni metod kojim prosirujemo niz naredbi novom naredbom, tj. u vektor dodajemo novi cvor */
    void Add(SyntaxTreeNode* stn);

    /* nasledjene metode */
    virtual void Print(std::ostream& s) const;
    virtual Funkcija* Interpret(FuncSymbolTable& fs) const;
    virtual double Calculate(FuncSymbolTable& fs, double x) const;
    virtual SyntaxTreeNode* Clone() const; 
};

/* izdvajamo novi tip koji predstavlja praznu naredbu */
class EmptyNode : public SyntaxTreeNode{

public:

    /* nasledjene metode */
    virtual void Print(std::ostream& s) const;
    virtual Funkcija* Interpret(FuncSymbolTable& fs) const;
    virtual double Calculate(FuncSymbolTable& fs, double x) const;
    virtual SyntaxTreeNode* Clone() const; 
};


#endif

